// Seleciona os botões que abrem a modal
const openModalButtons = document.querySelectorAll("[data-modal-target]");
// Seleciona os botões que fecham a modal
const closeModalButtons = document.querySelectorAll(".gmodal .bi-x");
// Cria um novo elemento div que funcionará como overlay
const overlay = document.createElement("div");
overlay.id = "modal_overlay"; // Define um ID único para o overlay
overlay.classList.add("modal_overlay"); // Adiciona classe ao overlay

// Adiciona evento de clique para abrir a modal em cada botão
openModalButtons.forEach((button) => {
  button.addEventListener("click", () => {
    const modalId = button.dataset.modalTarget;
    const modal = document.getElementById(modalId);
    openModal(modal);
  });
});

// Adiciona evento de clique para fechar a modal em cada botão
closeModalButtons.forEach((button) => {
  button.addEventListener("click", () => {
    const modal = button.closest(".gmodal");
    closeModal(modal);
  });
});

// Adiciona evento de clique no overlay para fechar a modal
overlay.addEventListener("click", closeModalOnOverlayClick);
// Adiciona evento para fechar a modal com a tecla ESC
document.addEventListener("keydown", closeModalOnEscKey);

// Função para abrir a modal e adicionar o overlay
function openModal(modal) {
  if (modal) {
    document.body.classList.add("modal-open");
    document.body.appendChild(overlay);
    modal.classList.add("open");
    setTimeout(() => {
      modal.classList.add("fade");
    }, 10);
    modal.classList.remove("closeModals");
  } else {
    console.error("Modal é nulo ou indefinido.");
  }
}

// Função para fechar a modal e remover o overlay
function closeModal(modal) {
  modal.classList.remove("open", "fade"); // Remove as classes 'open' e 'fade'
  modal.style.display = "none"; // Esconde a modal
  document.getElementById("modal_overlay").remove(); // Remove o overlay do body
}

// Função chamada ao clicar no overlay para fechar a modal
function closeModalOnOverlayClick() {
  const openModal = document.querySelector(".gmodal.open");
  if (openModal) {
    closeModal(openModal);
  }
}

// Função para fechar a modal com a tecla ESC
function closeModalOnEscKey(e) {
  if (e.key === "Escape") {
    const openModal = document.querySelector(".gmodal.open");
    if (openModal) {
      closeModal(openModal);
    }
  }
}

// Configura listeners para PJAX, se necessário
function setupModalListeners() {
  $(document).on("pjax:end pjax:success pjax:complete", () => {
    const openModal = $(".gmodal.open");
    if (openModal.length) {
      closeModal(openModal[0]);
    }
    setTimeout(function () {
      initializeModalListeners();
    }, 1000);
  });
}

// Inicializa os listeners dos botões de abrir e fechar a modal
function initializeModalListeners() {
  $("[data-modal-target]")
    .off("click")
    .on("click", function () {
      const modalId = $(this).data("modal-target");
      const modal = $("#" + modalId);
      openModal(modal[0]);
    });

  $(".gmodal .bi-x")
    .off("click")
    .on("click", function () {
      const modal = $(this).closest(".gmodal")[0];
      closeModal(modal);
    });
}

$(document).ready(setupModalListeners);

$(document).on("pjax:end", function () {
  verificarModalAutoShow();
});

// Função para verificar e exibir a modal com a classe 'show'
function verificarModalAutoShow() {
  const autoShowModal = document.querySelector(".gmodal.open");
  if (autoShowModal) {
    createOverlay(); // Cria o overlay
    showModal(autoShowModal); // Exibe a modal
  }
}

// Função para criar o overlay dinamicamente
function createOverlay() {
  const overlayExists = document.getElementById("modal_overlay");
  if (!overlayExists) {
    const overlay = document.createElement("div");
    overlay.id = "modal_overlay";
    overlay.classList.add("modal_overlay");
    document.body.insertBefore(overlay, document.body.lastChild);
  }
}

// Função para exibir a modal automaticamente
function showModal(modal) {
  document.body.classList.add("modal-open"); // Adiciona uma classe no body para indicar que a modal está aberta
  modal.classList.add("fade"); // Adiciona a classe 'fade' para animação
  setTimeout(() => {
    modal.classList.add("open"); // Adiciona a classe 'show' para exibir a modal
  }, 10);
}

// Adiciona evento de clique no overlay para fechar a modal
document.addEventListener("click", function (event) {
  if (event.target.id === "modal_overlay") {
    closeModal(document.querySelector(".gmodal.open"));
  }
});

// Função para fechar a modal
function closeModal(modal) {
  if (modal) {
    document.body.classList.remove("modal-open"); // Remove a classe do body
    modal.classList.remove("open", "fade"); // Remove as classes da modal
    modal.classList.add("closeModals"); // Esconde a modal

    removeOverlay(); // Remove o overlay do body
  }
}

// Função para remover o overlay
function removeOverlay() {
  const overlay = document.getElementById("modal_overlay");
  if (overlay) {
    overlay.remove();
  }
}

// Inicializa a verificação da modal no carregamento da página
document.addEventListener("DOMContentLoaded", verificarModalAutoShow);

// Verifica se a modal deve ser exibida com base no valor do cookie
function verificarExibicaoModal() {
  // Função para obter o cookie pelo nome
  function getCookie(nome) {
    // Divide a string do cookie e obtém todos os pares nome=valor em um array
    const arrCookie = document.cookie.split(";");
    // Percorre os elementos do array
    for (let i = 0; i < arrCookie.length; i++) {
      let parCookie = arrCookie[i].split("=");
      // Remove espaços em branco no início do nome do cookie
      // e compara com a string fornecida
      if (nome === parCookie[0].trim()) {
        // Decodifica o valor do cookie e retorna
        return decodeURIComponent(parCookie[1]);
      }
    }
    // Retorna null se não for encontrado
    return null;
  }

  const ultimaExibicao = getCookie("modalUltimaExibicao");
  if (ultimaExibicao) {
    const tempoAtual = new Date().getTime();
    const umaHora = 60 * 60 * 1000; // milissegundos
    // Verifica se passou uma hora desde a última exibição
    return tempoAtual - ultimaExibicao > umaHora;
  }
  // Se o cookie não existir, significa que a modal ainda não foi exibida
  return true;
}

// Função para definir o tempo de exibição do cookie
function setarTempoExibicaoModal() {
  const tempoAtual = new Date().getTime();
  // Define o cookie com expiração de 1 hora a partir de agora
  document.cookie =
    "modalUltimaExibicao=" + tempoAtual + ";max-age=3600;path=/";
}

// Função para abrir a modal
function abrirModal() {
  if (verificarExibicaoModal()) {
    let slidesModal = document.getElementById("slides-modal");

    if (slidesModal) {
      slidesModal.classList.add("open");
      createOverlay();
    }
    setarTempoExibicaoModal();
  }
}

// Adiciona evento listener para fechar a modal
document.addEventListener("DOMContentLoaded", function () {
  const iconeFechar = document.querySelector("#slides-modal .bi-x");
  if (iconeFechar) {
    iconeFechar.addEventListener("click", function () {
      document.getElementById("slides-modal").classList.remove("open");
    });
  }
});

// Chamada inicial para abrir a modal quando a página é carregada
document.addEventListener("DOMContentLoaded", abrirModal);
