window.Morpheus.lootbox = {};

async function createWheel() {
    const canvas = document.getElementById('lootbox-wheel-canvas');


    if (canvas) {
        const response = await fetch(`${window.Morpheus.basePath}lootboxes/${canvas.dataset.slug}/items`);
        const data = await response.json();


        const items = [];
        for (let i = 0, len = data.items.length; i < len; i++) {
            const item = data.items[i];
            items.push({
                label: item.name,
                image: item.image,
                color: item.rarity_color
            })
        }

        const wheel = new LootboxWheel({
            canvas,
            items,
            loops: canvas.dataset.loops ?? 4,
            duration: canvas.dataset.duration ?? 6000
        });

        const openButtonWheel = document.querySelector('[data-open-wheel]');
        if (openButtonWheel) {
            openButtonWheel.removeAttribute('disabled');
            openButtonWheel.addEventListener('click', (e) => {
                e.preventDefault();

                window.Morpheus.dialog.confirm(openButtonWheel.dataset.confirmation, async result => {
                    if (result.isConfirmed) {
                        try {
                            const response = await fetch(openButtonWheel.getAttribute('href'), {
                                headers: {
                                    'Accept': 'application/json'
                                }
                            });
                            const data = await response.json();

                            if (data.success) {
                                wheel.spinTo(data.index, () => {
                                    window.Morpheus.dialog.success(data.message);
                                });
                            } else {
                                window.Morpheus.dialog.error(data.message);
                            }
                        } catch (err) {
                            const error = typeof (err.error) === 'object' ? err.error : err;
                            const message = error.error ?? error.message ?? null;

                            window.Morpheus.dialog.error(message);
                        }
                    }
                })
            });
        }
    }
}

async function createCase() {
    const lootboxCase = document.getElementById('lootbox-case');
    if (lootboxCase) {
        const openButtonCase = document.querySelector('[data-open-case]');
        if (openButtonCase) {
            openButtonCase.removeAttribute('disabled');

            const lootbox = new LootboxCase(
                lootboxCase,
                {
                    loops: lootboxCase.dataset.loops ?? 4,
                    duration: lootboxCase.dataset.duration ?? 6000
                }
            )

            openButtonCase.addEventListener('click', (e) => {
                e.preventDefault();

                window.Morpheus.dialog.confirm(openButtonCase.dataset.confirmation, async result => {
                    if (result.isConfirmed) {
                        try {
                            const response = await fetch(openButtonCase.getAttribute('href'), {
                                headers: {
                                    'Accept': 'application/json'
                                }
                            });
                            const data = await response.json();

                            if (data.success) {
                                lootbox.spinTo(data.index, () => {
                                    window.Morpheus.dialog.success(data.message);
                                })
                            } else {
                                window.Morpheus.dialog.error(data.message);
                            }
                        } catch (err) {
                            const error = typeof (err.error) === 'object' ? err.error.message : err.message;

                            window.Morpheus.dialog.error(error);
                        }
                    }
                })
            });
        }
    }
}

class LootboxCase {
    constructor(root, { loops = 4, duration = 5000, tickVolume = 0.2, easing = 'cubic-bezier(0.08,0.7,0.12,1)' } = {}) {
        this.root = root;
        this.windowEl = root.querySelector('.lootbox-case__window');
        this.trackEl = root.querySelector('[data-lootbox-case-container]');
        this.itemSel = '[data-lootbox-case-item]';
        this.easing = easing;
        this.loops = loops;
        this.duration= duration;

        this.originals = Array.from(this.trackEl.querySelectorAll(this.itemSel));

        const first = this.originals[0];
        const cs = getComputedStyle(first);
        const marginLeft  = parseFloat(cs.marginLeft)  || 0;
        const marginRight = parseFloat(cs.marginRight) || 0;
        const gap = parseFloat(getComputedStyle(this.trackEl).columnGap || getComputedStyle(this.trackEl).gap || '0') || 0;

        this.cardW = first.getBoundingClientRect().width;
        this.step  = this.cardW + marginLeft + marginRight + gap;
        this.baseWidth = this.step * this.originals.length;

        this.repeatCount = 30;
        this.spinning = false;
        this.currentX = 0;

        this.audioCtx = null;
        this.tickVolume = tickVolume;
        this.tickerRunning = false;
        this.lastTickIndex = null;

        this.buildTrack();
        this.resetToIndex(0);

        window.addEventListener('resize', () => { if (!this.spinning) this.resetToIndex(this.indexFromCurrent()); });
    }

    buildTrack(){
        const originals = this.originals;
        const frag = document.createDocumentFragment();
        for (let r=0; r < this.repeatCount; r++) {
            originals.forEach(node => frag.appendChild(node.cloneNode(true)));
        }
        this.trackEl.innerHTML = '';
        this.trackEl.appendChild(frag);
    }

    centerOffset(){
        const winW = this.windowEl.getBoundingClientRect().width;
        return (winW/2) - (this.cardW/2);
    }

    clampIndex(i){
        const n = this.originals.length;
        return ((i % n) + n) % n;
    }

    setTransform(x, withTransition=false, duration=0){
        this.trackEl.style.transition = withTransition ? `transform ${duration}ms ${this.easing}` : 'none';
        this.trackEl.style.transform  = `translate3d(${x}px,0,0)`;
    }

    xForIndex(i) {
        const k = Math.floor(this.repeatCount / 2);
        return -(k * this.baseWidth + i * this.step) + this.centerOffset();
    }

    indexFromX(x) {
        const k = Math.floor(this.repeatCount / 2);
        const dist = -(x - this.centerOffset()) - k * this.baseWidth;
        const steps = Math.round(dist / this.step);
        return this.clampIndex(steps);
    }

    indexFromCurrent() {
        const k = Math.floor(this.repeatCount / 2);
        const dist = -(this.currentX - this.centerOffset()) - k * this.baseWidth;
        const steps = Math.round(dist / this.step);
        return this.clampIndex(steps);
    }

    resetToIndex(i){
        this.currentX = this.xForIndex(this.clampIndex(i));
        this.setTransform(this.currentX, false);
    }

    ensureAudio(){
        if (!this.audioCtx) this.audioCtx = new (window.AudioContext || window.webkitAudioContext)();
    }

    readCurrentXFromTransform() {
        const style = getComputedStyle(this.trackEl);
        const t = style.transform || 'none';
        if (t === 'none') return this.currentX;
        if (t.startsWith('matrix3d(')) {
            const vals = t.slice(9, -1).split(',').map(parseFloat);
            return vals[12] || 0;
        }
        if (t.startsWith('matrix(')) {
            const vals = t.slice(7, -1).split(',').map(parseFloat);
            return vals[4] || 0;
        }
        return this.currentX;
    }

    startTicker(){
        this.tickerRunning = true;
        this.lastTickIndex = null;
        const loop = () => {
            if (!this.tickerRunning) return;
            const x = this.readCurrentXFromTransform();
            const idx = this.indexFromX(x);
            if (idx !== this.lastTickIndex){
                this.lastTickIndex = idx;
                this.playTick();
            }
            requestAnimationFrame(loop);
        };

        requestAnimationFrame(loop);
    }

    stopTicker(){
        this.tickerRunning = false;
    }

    playTick(){
        // beep curto (~30ms)
        this.ensureAudio();
        const ctx = this.audioCtx;
        const osc = ctx.createOscillator();
        const gain = ctx.createGain();
        osc.type = 'square';
        osc.frequency.value = 1100;
        gain.gain.value = this.tickVolume;
        osc.connect(gain).connect(ctx.destination);
        const now = ctx.currentTime;
        osc.start(now);
        gain.gain.setValueAtTime(this.tickVolume, now);
        gain.gain.exponentialRampToValueAtTime(0.001, now + 0.03);
        osc.stop(now + 0.035);
    }

    spinTo(targetIndex, onComplete) {
        if (this.spinning) return;
        this.spinning = true;

        this.resetToIndex(this.indexFromCurrent());

        const startIdx = this.indexFromCurrent();
        let delta = targetIndex - startIdx;
        if (delta <= 0) delta += this.originals.length;

        const distance = (this.loops * this.baseWidth) + (delta * this.step);

        this.startTicker();

        requestAnimationFrame(() => {
            this.setTransform(this.currentX - distance, true, this.duration);
        });

        const onEnd = () => {
            this.trackEl.removeEventListener('transitionend', onEnd);
            this.stopTicker();
            this.resetToIndex(targetIndex);
            this.spinning = false;

            onComplete && onComplete()
        };

        this.trackEl.addEventListener('transitionend', onEnd);
    }
}

class LootboxWheel  {
    constructor({ canvas, items, loops = 6, duration = 5000}) {
        this.canvas = canvas;
        this.ctx = canvas.getContext('2d');
        this.items = Array.isArray(items) ? items : [];
        this.loops = Math.max(0, Math.round(loops));
        this.duration = duration;
        this.centerX = canvas.width / 2;
        this.centerY = canvas.height / 2;
        this.radius = Math.min(canvas.width, canvas.height) * 0.47;
        this.innerRadius = this.radius * 0.20;
        this.imageRadius = this.radius * 0.78;
        this.imageMaxSize = Math.floor(this.radius * 0.30);
        this.pointerAngle = -Math.PI / 2;
        this.sliceAngle = (2 * Math.PI) / Math.max(1, this.items.length);
        this.angle = 0;
        this.animating = false;
        this.animStart = 0;
        this.animDuration = 0;
        this.animFrom = 0;
        this.animTo = 0;
        this.aimOffsetRad = 0;
        this.epsilon = 1e-9;
        this.audioContext = null;
        this.lastTickSlice = null;
        this.images = [];
        this.onComplete = null;
        this.targetIndexActive = null;
        this.loadImages();
        this.loop = this.loop.bind(this);
        requestAnimationFrame(this.loop);
    }

    loadImages() {
        const jobs = this.items.map((item, i) => {
            const img = new Image();
            img.crossOrigin = 'anonymous';
            this.images[i] = img;
            return new Promise(resolve => {
                img.onload = () => resolve(true);
                img.onerror = () => resolve(false);
                if (item.image) {
                    img.src = item.image;
                } else {
                    resolve(false);
                }
            });
        });
        Promise.all(jobs).then(() => {});
    }

    spinTo(targetIndex, onComplete) {
        const tau = Math.PI * 2;
        const total = this.items.length;
        const idx = Number.isFinite(targetIndex) ? this.clamp(Math.round(targetIndex), 0, Math.max(0, total - 1)) : 0;
        this.angle = 0;
        const center = this.indexToCenterAngle(idx) + this.aimOffsetRad;
        let delta = this.pointerAngle - center;
        delta = ((delta % tau) + tau) % tau;
        this.animFrom = 0;
        this.animTo = this.loops * tau + delta;
        this.animStart = performance.now();
        this.animDuration = this.duration;
        this.animating = true;
        this.onComplete = typeof onComplete === 'function' ? onComplete : null;
        this.targetIndexActive = idx;
    }

    stopSpinHard() { this.animating = false; this.animTo = this.angle; }

    loop(ts) {
        if (!this.prev) { this.prev = ts; }
        this.prev = ts;
        if (this.animating) { this.updateAnimation(ts); this.handleTicks(); }
        this.draw();
        requestAnimationFrame(this.loop);
    }

    updateAnimation(ts) {
        const t = this.clamp((ts - this.animStart) / this.animDuration, 0, 1);
        const eased = this.easeOutQuint(t);
        this.angle = this.animFrom + (this.animTo - this.animFrom) * eased;
        if (t >= 1) {
            this.animating = false;
            this.angle = this.animTo;
            const doneIndex = this.targetIndexActive;
            const doneItem = doneIndex != null ? this.items[doneIndex] : null;
            if (this.onComplete) { this.onComplete({ index: doneIndex, item: doneItem }); }
            this.onComplete = null;
            this.targetIndexActive = null;
        }
    }

    handleTicks() {
        const atPointer = this.normalizeAngle(this.pointerAngle - this.angle + this.epsilon);
        const idx = this.angleToIndex(atPointer);
        if (idx !== this.lastTickSlice) { this.playTick(); this.lastTickSlice = idx; }
    }

    draw() {
        const ctx = this.ctx;
        ctx.clearRect(0, 0, this.canvas.width, this.canvas.height);
        ctx.save();
        ctx.translate(this.centerX, this.centerY);
        ctx.rotate(this.angle);
        this.drawWheel();
        ctx.restore();
        this.drawPointerTop();
    }

    drawWheel() {
        for (let i = 0; i < this.items.length; i++) {
            const start = i * this.sliceAngle;
            const end = start + this.sliceAngle;
            this.drawSliceBackgroundByColor(this.items[i].color, start, end);
            this.drawSliceImage(i, start, end);
            this.drawSliceSeparator(start);
        }
        this.drawCenter();
        this.drawOuterRim();
    }

    drawOuterRim() {
        const ctx = this.ctx;
        ctx.beginPath();
        ctx.arc(0, 0, this.radius, 0, 2 * Math.PI);
        ctx.lineWidth = 1;
        ctx.strokeStyle = '#000';
        ctx.stroke();
    }

    drawSliceBackgroundByColor(baseColor, start, end) {
        const ctx = this.ctx;
        const hsl = this.parseColor(baseColor);
        const c1 = this.toCss(this.adjustLightness(hsl, 0.25));
        const c2 = this.toCss(this.adjustLightness(hsl, -0.18));
        const mid = (start + end) / 2;
        const x1 = Math.cos(mid) * this.innerRadius;
        const y1 = Math.sin(mid) * this.innerRadius;
        const x2 = Math.cos(mid) * this.radius;
        const y2 = Math.sin(mid) * this.radius;
        const g = ctx.createLinearGradient(x1, y1, x2, y2);
        g.addColorStop(0, c1);
        g.addColorStop(1, c2);
        ctx.beginPath();
        ctx.moveTo(0, 0);
        ctx.arc(0, 0, this.radius, start, end, false);
        ctx.closePath();
        ctx.fillStyle = g;
        ctx.fill();
    }

    drawSliceSeparator(angle) {
        const ctx = this.ctx;
        ctx.save();
        ctx.rotate(angle);
        ctx.beginPath();
        ctx.moveTo(0, 0);
        ctx.lineTo(this.radius, 0);
        ctx.lineWidth = 1;
        ctx.lineCap = 'butt';
        ctx.strokeStyle = 'rgba(0,0,0,0.45)';
        ctx.stroke();
        ctx.restore();
    }

    drawSliceImage(i, start, end) {
        const img = this.images[i];
        if (!img || !img.complete) return;

        const angle = (start + end) / 2;
        const maxS = this.imageMaxSize;
        const scale = Math.min(1, maxS / Math.max(img.width, img.height));
        const w = Math.floor(img.width * scale);
        const h = Math.floor(img.height * scale);

        const r = this.imageRadius * 0.85;

        const ctx = this.ctx;
        ctx.save();
        ctx.translate(Math.cos(angle) * r, Math.sin(angle) * r);
        ctx.rotate(angle + Math.PI / 2);
        ctx.drawImage(img, -w / 2, -h, w, h);
        ctx.restore();
    }

    drawCenter() {
        // const ctx = this.ctx;
        // ctx.beginPath();
        // ctx.arc(0, 0, 10, 0, 2 * Math.PI);
        // ctx.fillStyle = '#fff';
        // ctx.fill();
        // ctx.lineWidth = 2;
        // ctx.strokeStyle = 'rgba(255,255,255,0.06)';
        // ctx.stroke();

        const ctx = this.ctx;
        ctx.beginPath();
        ctx.arc(0, 0, 10, 0, 2 * Math.PI);
        ctx.fillStyle = '#fff';
        ctx.fill();
        ctx.lineWidth = 1;
        ctx.strokeStyle = '#000';
        ctx.stroke();
    }

    drawPointerTop() {
        const ctx = this.ctx;
        const baseY = this.centerY - this.radius - 10;
        const halfWidth = 15;
        const height = 25;

        ctx.save();
        ctx.translate(this.centerX, baseY);
        ctx.beginPath();
        ctx.moveTo(-halfWidth, 0);
        ctx.lineTo(halfWidth, 0);
        ctx.lineTo(0, height);
        ctx.closePath();
        ctx.lineWidth = 2;
        ctx.lineJoin = 'round';
        ctx.strokeStyle = '#000';
        ctx.fillStyle = '#fff';
        ctx.stroke();
        ctx.fill();
        ctx.restore();
    }


    angleToIndex(angle) {
        const a = this.normalizeAngle(angle);
        let idx = Math.floor((a + this.epsilon) / this.sliceAngle);
        if (idx >= this.items.length) { idx = this.items.length - 1; }
        if (idx < 0) { idx = 0; }
        return idx;
    }

    indexToCenterAngle(index) {
        const start = index * this.sliceAngle;
        const center = start + this.sliceAngle / 2;
        return this.normalizeAngle(center);
    }

    normalizeAngle(a) {
        let x = a;
        const tau = Math.PI * 2;
        while (x < 0) { x += tau; }
        while (x >= tau) { x -= tau; }
        return x;
    }

    easeOutQuint(t) {
        const x = 1 - t;
        return 1 - x * x * x * x * x;
    }

    playTick() {
        try {
            if (!this.audioContext) { this.audioContext = new (window.AudioContext || window.webkitAudioContext)(); }
            const ctx = this.audioContext;
            const o = ctx.createOscillator();
            const g = ctx.createGain();
            o.type = 'square';
            o.frequency.value = 900;
            g.gain.value = 0.06;
            o.connect(g);
            g.connect(ctx.destination);
            o.start();
            o.stop(ctx.currentTime + 0.04);
        } catch (e) {}
    }

    clamp(v, min, max) {
        if (v < min) {
            return min;
        } if (v > max) {
            return max;
        }

        return v;
    }

    parseColor(css) {
        const c = document.createElement('canvas');
        const x = c.getContext('2d');
        x.fillStyle = css || '#888';
        const computed = x.fillStyle;
        const d = document.createElement('div');
        d.style.color = computed;
        document.body.appendChild(d);
        const rgb = getComputedStyle(d).color;
        document.body.removeChild(d);
        const m = rgb.match(/rgba?\((\d+),\s*(\d+),\s*(\d+)/i);
        const r = m ? parseInt(m[1], 10) : 128;
        const g = m ? parseInt(m[2], 10) : 128;
        const b = m ? parseInt(m[3], 10) : 128;
        return this.rgbToHsl(r, g, b);
    }

    adjustLightness(hsl, delta) {
        const h = hsl.h, s = hsl.s, l = Math.max(0, Math.min(1, hsl.l + delta));
        return { h, s, l };
    }

    toCss(hsl) {
        const [r, g, b] = this.hslToRgb(hsl.h, hsl.s, hsl.l);
        return `rgb(${Math.round(r)},${Math.round(g)},${Math.round(b)})`;
    }

    rgbToHsl(r, g, b) {
        const rn = r / 255, gn = g / 255, bn = b / 255;
        const max = Math.max(rn, gn, bn), min = Math.min(rn, gn, bn);
        let h = 0, s = 0, l = (max + min) / 2;
        if (max !== min) {
            const d = max - min;
            s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
            switch (max) {
                case rn: h = (gn - bn) / d + (gn < bn ? 6 : 0); break;
                case gn: h = (bn - rn) / d + 2; break;
                case bn: h = (rn - gn) / d + 4; break;
            }
            h /= 6;
        }
        return { h, s, l };
    }

    hslToRgb(h, s, l) {
        if (s === 0) { const v = l * 255; return [v, v, v]; }
        const q = l < 0.5 ? l * (1 + s) : l + s - l * s;
        const p = 2 * l - q;
        return [
            255 * this.hue2rgb(p, q, h + 1/3),
            255 * this.hue2rgb(p, q, h),
            255 * this.hue2rgb(p, q, h - 1/3)
        ];
    }

    hue2rgb(p, q, t) {
        if (t < 0) t += 1;
        if (t > 1) t -= 1;
        if (t < 1/6) return p + (q - p) * 6 * t;
        if (t < 1/2) return q;
        if (t < 2/3) return p + (q - p) * (2/3 - t) * 6;
        return p;
    }
}



DOMLoaded(() => {
    createWheel();
    createCase();
})