/**
 * =================================================================================
 * OVERRIDE PARA O PAINEL DE EVENTOS EM DESTAQUE (SPOTLIGHT) - v3
 * =================================================================================
 * Este script modifica o comportamento padrão do `events.js` para a área de
 * spotlight, incluindo:
 * 1. Estrutura de lista (ul/li).
 * 2. Remoção do prefixo de horário do nome do evento.
 * 3. Destaque para eventos que começam em 5 minutos ou menos (classe 'bg-success').
 * =================================================================================
 */

document.addEventListener("DOMContentLoaded", function () {
  if (
    typeof renderEvents !== "function" ||
    typeof updateEventsTime !== "function"
  ) {
    console.error(
      "Erro: As funções 'renderEvents' ou 'updateEventsTime' não foram encontradas. Certifique-se de que events.js foi carregado antes deste script."
    );
    return;
  }

  const originalRenderEvents = renderEvents;
  const originalUpdateEventsTime = updateEventsTime;

  /**
   * @override renderEvents
   */
  renderEvents = function (el, events, timezone) {
    const isSpotlight = el.dataset?.spotlight !== undefined;

    if (!isSpotlight) {
      originalRenderEvents(el, events, timezone);
      return;
    }

    const listItemTemplate = [
      // Adicionamos uma classe ao span para facilitar a seleção
      '<li class="list-group-item" id="{id}">',
      "  <p>{name}</p>",
      '  <span class="badge__text events-event-time"></span>',
      "</li>",
    ].join("");

    const htmlByCategory = {};

    for (let i = 0; i < events.length; i++) {
      const event = events[i];
      if (!event.spotlight) continue;

      const categoryId = event.category_id === null ? "*" : event.category_id;
      if (htmlByCategory[categoryId] === undefined) {
        htmlByCategory[categoryId] = [];
      }

      htmlByCategory[categoryId].push(
        listItemTemplate
          .replace("{id}", `event-spotlight-${event.id}`)
          .replace("{name}", event.name)
      );
    }

    for (let categoryId in htmlByCategory) {
      const container = el.querySelector(`.events-${categoryId}`);
      if (container) {
        container.innerHTML = `<ul class="list-group list-group-flush mt-3">${htmlByCategory[
          categoryId
        ].join("")}</ul>`;
      }
    }

    updateEventsTime(el, events, timezone);
  };

  /**
   * @override updateEventsTime
   */
  updateEventsTime = function (el, events, timezone) {
    const isSpotlight = el.dataset?.spotlight !== undefined;

    if (!isSpotlight) {
      originalUpdateEventsTime(el, events, timezone);
      return;
    }

    const date = toDateUTC(timezone);
    const time = toSeconds(
      date.getHours(),
      date.getMinutes(),
      date.getSeconds()
    );

    for (let i = 0; i < events.length; i++) {
      const event = events[i];
      if (!event.spotlight) continue;

      const eventElement = document.getElementById(
        `event-spotlight-${event.id}`
      );
      if (!eventElement) continue;

      if (event.type === 1) {
        let nextEventTime = null;
        let nextTimeInSeconds = null;

        for (let j = 0; j < event.times.length; j++) {
          const eventTimeStr = event.times[j];
          const parts = eventTimeStr.split(":");
          const t = toSeconds(parseInt(parts[0]), parseInt(parts[1]), 0);

          if (t > time) {
            nextEventTime = eventTimeStr;
            nextTimeInSeconds = t;
            break;
          }
        }

        if (nextEventTime === null) {
          nextEventTime = event.times[0];
          const parts = nextEventTime.split(":");
          nextTimeInSeconds = toSeconds(
            parseInt(parts[0]),
            parseInt(parts[1]),
            0
          );
        }

        let diff = nextTimeInSeconds - time;
        if (diff < 0) diff += 3600 * 24;

        const hour = Math.floor(diff / 3600);
        const minute = Math.floor((diff % 3600) / 60);
        const second = diff % 60;
        const countdown = `${("0" + hour).slice(-2)}:${("0" + minute).slice(
          -2
        )}:${("0" + second).slice(-2)}`;

        const nameElement = eventElement.querySelector("p");
        const timeElement = eventElement.querySelector(".events-event-time");

        if (nameElement) nameElement.textContent = event.name;
        if (timeElement) timeElement.textContent = countdown;

        // <-- INÍCIO DA MUDANÇA -->
        // Lógica para adicionar/remover a classe 'bg-success'
        if (timeElement) {
          // Condição: 5 minutos (300 segundos) ou menos
          if (diff <= 300) {
            timeElement.classList.add("bg-success");
          } else {
            timeElement.classList.remove("bg-success");
          }
        }
        // <-- FIM DA MUDANÇA -->
      }
    }
  };
});
