function toSeconds(hour, minute, second) {
    return hour * 3600 + minute * 60 + second;
}

function toDateUTC(hours) {
    const d = new Date();
    const utc = d.getTime() + (d.getTimezoneOffset() * 60000);

    return new Date(utc + (3600000 * hours));
}

function getSavedEvents() {
    let events = window.localStorage.getItem('events');

    return !events ? [] : events.split(',');
}

function renderCheckbox(event, spotlight) {
    const savedEvents = getSavedEvents();
    const checked = savedEvents.indexOf(event.id.toString()) >= 0;
    const template = [
        '<div class="form-check form-switch">',
        `<input class="form-check-input events-event-notify-switch" id="event-${spotlight ? 'spotlight-' : ''}${event.id}" type="checkbox" value="${event.id}"${checked ? ' checked' : ''} />`,
        '</div>'
    ].join('');

    if (!spotlight) {
        return template;
    }

    return '';
}

function removeElement(arr) {
    let what, a = arguments, L = a.length, ax;
    while (L > 1 && arr.length) {
        what = a[--L];
        while ((ax= arr.indexOf(what)) !== -1) {
            arr.splice(ax, 1);
        }
    }
    return arr;
}

function renderEvents(el, events, timezone) {
    const spotlight = el.dataset?.spotlight !== undefined;
    const template = [
        '<div class="events-event" id="{id}">',
        '<div class="events-event-name">',
        '{checkbox} <span>{name}</span>',
        '</div>',
        '<div class="events-event-time"></div>',
        '</div>'
    ].join('');

    const html = {};
    for (let i = 0; i < events.length; i++) {
        const event = events[i];

        if (spotlight && !event.spotlight) continue;

        const category = event.category_id === null ? '*' : event.category_id;
        if (html[category] === undefined) {
            html[category] = [];
        }

        html[category].push(template.replace('{checkbox}', event.type === 3 ? '' : renderCheckbox(event, spotlight))
            .replace('{name}', event.name)
            .replace('{id}', `event-${spotlight ? 'spotlight-' : ''}${event.id}`));
    }

    for (let category in html) {
        const container = el.querySelector('.events' + (category === '*' ? '' : '-' + category));
        if (container) {
            container.innerHTML = html[category].join('');
        }
    }

    updateEventsTime(el, events, timezone);
}

function updateEventsTime(el, events, timezone) {
    const spotlight = el.dataset?.spotlight !== undefined;
    const date = toDateUTC(timezone);
    const time = toSeconds(date.getHours(), date.getMinutes(), date.getSeconds());

    const classDefault = 'badge bg-secondary';
    const classSuccess = 'badge bg-success';

    const html = {};
    for (let i = 0; i < events.length; i++) {
        const event = events[i];
        const eventElement = document.getElementById(`event-${spotlight ? 'spotlight-' : ''}${event.id}`)
        let df = -1;

        if (event.type === 1) {
            let nextEvent = null;
            let nextTime = null;
            let timesMap = {};
            for (let j = 0; j < event.times.length; j++) {
                const eventTime = event.times[j];
                const parts = eventTime.split(':');
                const t = toSeconds(parts[0], parts[1], 0);

                timesMap[t] = eventTime;

                if (t > time) {
                    nextEvent = eventTime;
                    nextTime = t;
                    break;
                }
            }

            if (nextEvent === null) {
                nextEvent = Object.values(timesMap)[0];
                nextTime = Object.keys(timesMap)[0];
            }

            let diff = nextTime - time;
            if (diff < 0) diff += 3600 * 24;

            df = diff;

            let klass = classDefault;
            if (diff < 15 * 60) klass = classSuccess;

            const hour = parseInt(diff / 3600);
            diff -= 3600 * hour;
            const minute = parseInt(diff / 60);
            const second = diff - minute * 60;
            const countdown = ('0' + hour).slice(-2) + ':' + ('0' + minute).slice(-2) + ':' + ('0' + second).slice(-2);

            const nameElement = eventElement.querySelector('.events-event-name span');
            nameElement.innerHTML = `${nextEvent} ${event.name}`;

            const timeElement = eventElement.querySelector('.events-event-time');
            timeElement.innerHTML = `<span class="badge ${klass}">${countdown}</span>`;
        } else if (event.type === 2) {
            let week = date.getDay();
            let counter = 0;
            let next = [0, 0, 0, 0, 0, 0, 0];
            let hoursText = [];

            for (let day in event.times) {
                const hrs = event.times[day];
                const w = day - 1;

                for (let h in hrs) {
                    let hour = hrs[h].split(':');
                    let ht = 0;
                    if (hour[0] === '00' && hour[1] === '00') {
                        ht = toSeconds(hour[0], hour[1], 1);
                    } else {
                        ht = toSeconds(hour[0], hour[1], 0);
                    }

                    if (next[w] === 0 || time - ht <= 0) {
                        next[w] = ht;
                        hoursText[w] = hrs;

                        break;
                    }
                }
            }

            for (; 1;) {
                if (next[week]
                    && (
                        week === date.getDay()
                        && next[week] > time
                        || week !== date.getDay()
                    )
                ) {
                    let dayOfWeek = window.Morpheus.language.events.weekDays[week];
                    if (week === date.getDay()) {
                        dayOfWeek = '';
                    }

                    let diff = week * 24 * 60 * 60 + next[week] - (date.getDay() * 24 * 60 * 60 + time);
                    if (diff < 0) {
                        diff = diff + 7 * 24 * 60 * 60;
                    }

                    df = diff;

                    let klass = classDefault;
                    if (diff < 60 * 60) klass = classSuccess;

                    const ht = parseInt(diff / (24 * 60 * 60));
                    diff = diff - ht * (24 * 60 * 60);
                    const hour = parseInt(diff / 3600);
                    diff = diff - 3600 * hour;
                    const minute = parseInt(diff / 60);
                    const second = diff - minute * 60;

                    let text = '';
                    if (ht) {
                        text = ht + "d " + (hour === 0 ? minute + 'm' : hour + 'h');
                    } else {
                        text = ('0' + hour).slice(-2) + ':' + ('0' + minute).slice(-2) + ':' + ('0' + second).slice(-2);
                    }

                    const nameElement = eventElement.querySelector('.events-event-name span');
                    nameElement.innerHTML = `${dayOfWeek} ${hoursText[week]} ${event.name}`;

                    const timeElement = eventElement.querySelector('.events-event-time');
                    timeElement.innerHTML = `<span class="badge ${klass}">${text}</span>`;

                    break;
                }
                if (counter++ > 8) {
                    const nameElement = eventElement.querySelector('.events-event-name span');
                    nameElement.innerHTML = event.name;

                    const timeElement = eventElement.querySelector('.events-event-time');
                    timeElement.innerHTML = `<span class="badge ${classDefault}">7d</span>`;

                    break;
                }
                week = (week + 1) % 7;
            }
        } else {
            const nameElement = eventElement.querySelector('.events-event-name span');
            nameElement.innerHTML = event.name;

            const timeElement = eventElement.querySelector('.events-event-time');
            timeElement.innerHTML = `<span>${event.times}</span>`;
        }

        const savedEvents = getSavedEvents();
        if (df === 60 * 5 && savedEvents.includes(event.id)) {
            notifyMe(printf(window.Morpheus.language.events.notifyTitle, event.name), printf(window.Morpheus.language.events.notifyBody, event.name));
        }
    }
}

function notifyMe(title, body) {
    if (!window.Notification) {
        return;
    }
    if (Notification.permission !== 'granted') {
        Notification.requestPermission();
    } else {
        const n = new Notification(title, {
            icon: window.Morpheus.logo,
            body: body
        });
        n.onclick = () => window.focus();
    }
}

DOMLoaded(() => {
    if (window.Notification && Notification.permission !== 'granted') {
        Notification.requestPermission();
    }

    const categories = document.querySelectorAll('[data-events-category]');
    categories.forEach(el => {
        let eventName = 'click';
        if (el.tagName === 'SELECT') {
            eventName = 'change';
        }

        el.addEventListener(eventName, e => {
            let value = el.dataset.value;
            if (el.tagName === 'SELECT') {
                value = el.value;
            }
            const container = el.closest('[data-events-area]');
            const events = container.querySelectorAll('.events');
            events.forEach(el2 => el2.style.display = 'none');

            if (value === '*') {
                events.forEach(el2 => el2.style.display = 'block');
            } else {
                const event = container.querySelector(`.events-${value}`);
                event.style.display = 'block';
            }
        });
    });

    (async () => {
        const response = await fetch(`${window.Morpheus.basePath}events?json=true`)
        const json = await response.json();
        const events = json.events;
        const eventsArea = document.querySelectorAll('[data-events-area]');

        eventsArea.forEach(el => {
            renderEvents(el, events, json.timezone);
            setInterval(() => updateEventsTime(el, events, json.timezone), 1000);

            const switches = document.querySelectorAll('.events-event-notify-switch');
            switches.forEach(el2 => {
                el2.addEventListener('click', e => {
                    let savedEvents = getSavedEvents();
                    if (e.target.checked && !savedEvents.includes(e.target.value)) {
                        savedEvents.push(e.target.value);
                    } else {
                        savedEvents = removeElement(savedEvents, e.target.value);
                    }

                    window.localStorage.setItem('events', savedEvents.join(','));
                });
            });
        });
    })()

});