class CustomSlider {
  constructor(selector, options = {}) {
    this.container = document.querySelector(selector);
    if (!this.container) {
      return;
    }
    this.slider = this.container.querySelector(".character-slots");
    this.items = Array.from(this.slider.children);
    this.prevButton = document.querySelector(".nav-arrow.prev");
    this.nextButton = document.querySelector(".nav-arrow.next");

    if (this.items.length === 0) return;

    this.settings = {
      centerMode: true,
      infinite: true,
      autoplay: true,
      autoplayDelay: 3000,
      transitionSpeed: 500,
      spacing: 15,
      initialSlide: 0,
      ...options,
    };

    this.originalItemsCount = this.items.length;
    this.currentIndex = this.settings.initialSlide;
    this.isTransitioning = false;
    this.autoplayInterval = null;

    this.init();
  }

  init() {
    this.slider.style.gap = `${this.settings.spacing}px`;
    this.slider.style.transition = `transform ${
      this.settings.transitionSpeed / 1000
    }s ease-in-out`;

    if (this.settings.infinite) {
      this.cloneItems();
      this.currentIndex += this.originalItemsCount;
    }

    this.attachEvents();
    this.updateSlider(false);

    if (this.settings.autoplay) {
      this.startAutoplay();
    }
  }

  cloneItems() {
    for (let i = this.originalItemsCount - 1; i >= 0; i--) {
      const clone = this.items[i].cloneNode(true);
      clone.dataset.isClone = true;
      this.slider.insertBefore(clone, this.slider.firstChild);
    }
    for (let i = 0; i < this.originalItemsCount; i++) {
      const clone = this.items[i].cloneNode(true);
      clone.dataset.isClone = true;
      this.slider.appendChild(clone);
    }
    this.items = Array.from(this.slider.children);
  }

  attachEvents() {
    this.nextButton.addEventListener("click", () => this.moveToNext());
    this.prevButton.addEventListener("click", () => this.moveToPrev());

    this.container.addEventListener("mouseenter", () => this.stopAutoplay());
    this.container.addEventListener("mouseleave", () => this.startAutoplay());

    this.slider.addEventListener("transitionend", () =>
      this.handleTransitionEnd()
    );

    this.items.forEach((item, index) => {
      item.addEventListener("click", () => {
        if (item.dataset.isClone) return;
        this.goTo(index);
      });
    });
  }

  updateSlider(withTransition = true) {
    this.isTransitioning = true;

    this.items.forEach((item) => item.classList.remove("active"));
    const activeItem = this.items[this.currentIndex];
    if (activeItem) {
      activeItem.classList.add("active");
    }

    const itemWidth = this.items[0].offsetWidth;
    const containerWidth = this.container.offsetWidth;
    let offset = 0;

    if (this.settings.centerMode) {
      offset =
        containerWidth / 2 -
        itemWidth / 2 -
        this.currentIndex * (itemWidth + this.settings.spacing);
    } else {
      offset = -this.currentIndex * (itemWidth + this.settings.spacing);
    }

    this.slider.style.transition = withTransition
      ? `transform ${this.settings.transitionSpeed / 1000}s ease-in-out`
      : "none";
    this.slider.style.transform = `translateX(${offset}px)`;

    if (!withTransition) {
      setTimeout(() => {
        this.isTransitioning = false;
      }, 50);
    }
  }

  goTo(index) {
    if (this.isTransitioning) return;
    this.currentIndex = index;
    this.updateSlider();
  }

  moveToNext() {
    if (this.isTransitioning) return;
    this.currentIndex++;
    this.updateSlider();
  }

  moveToPrev() {
    if (this.isTransitioning) return;
    this.currentIndex--;
    this.updateSlider();
  }

  handleTransitionEnd() {
    this.isTransitioning = false;

    if (!this.settings.infinite) return;

    if (this.currentIndex >= this.originalItemsCount * 2) {
      this.currentIndex = this.currentIndex - this.originalItemsCount;
      this.updateSlider(false);
    } else if (this.currentIndex < this.originalItemsCount) {
      this.currentIndex = this.currentIndex + this.originalItemsCount;
      this.updateSlider(false);
    }
  }

  startAutoplay() {
    if (!this.settings.autoplay || this.autoplayInterval) return;
    this.autoplayInterval = setInterval(() => {
      this.moveToNext();
    }, this.settings.autoplayDelay);
  }

  stopAutoplay() {
    clearInterval(this.autoplayInterval);
    this.autoplayInterval = null;
  }
}
