window.Morpheus.lootbox = {};

const audioMp3 = new Audio(`${window.Morpheus.basePath}plugins/LootBox/assets/sounds/tick.mp3`);
const audioWav = new Audio(`${window.Morpheus.basePath}plugins/LootBox/assets/sounds/tick.wav`);

async function createWheel() {
    const canvas = document.getElementById('lootbox-wheel-canvas');

    if (canvas) {
        const response = await fetch(`${window.Morpheus.basePath}lootboxes/${canvas.dataset.slug}/items`);
        const data = await response.json();
        const colors =  window.Morpheus.lootbox.colors || [
            '#ccc',
            '#bbb',
            '#aaa'
        ];

        const segments = [];
        for (let i = 0; i < data.items.length; i++) {
            const item = data.items[i];
            segments.push({
                fillStyle: item.rarity_color ?? colors[i % colors.length],
                text: item.name + ' (' + item.item_percent + '%)',
                size: winwheelPercentToDegrees(item.item_percent)
            });
        }

        const wheel = new Winwheel({
            canvasId: 'lootbox-wheel-canvas',
            numSegments: data.items.length,
            outerRadius: 170,
            lineWidth: 2,
            textFontSize: 12,
            segments: segments,
            animation: {
                direction: 'anti-clockwise',
                type: 'spinToStop',
                duration: 5,
                spins: 8,
                soundTrigger: 'pin',
                callbackFinished: function (e) {
                    window.Morpheus.dialog.success(wheel.successMessage);
                },
                callbackAfter: drawTriangle,
                callbackSound: playSound
            },
            pins: {
                number: 24
            }
        });

        function playSound(e) {
            audioMp3.pause();
            audioMp3.currentTime = 0;
            audioMp3.play();
        }

        function resetWheel() {
            wheel.stopAnimation(false);
            wheel.rotationAngle = 0;
            wheel.draw();
        }

        drawTriangle();
        function drawTriangle()  {
            const ctx = wheel.ctx;

            ctx.strokeStyle = '#000';
            ctx.fillStyle = '#fff';
            ctx.lineWidth = 3;
            ctx.lineJoin = 'round';
            ctx.beginPath();
            ctx.moveTo(180, 5);
            ctx.lineTo(220, 5);
            ctx.lineTo(200, 40);
            ctx.lineTo(181, 5);
            ctx.stroke();
            ctx.fill();
        }

        const openButtonWheel = document.querySelector('[data-open-wheel]');
        if (openButtonWheel) {
            openButtonWheel.removeAttribute('disabled');
            openButtonWheel.addEventListener('click', (e) => {
                e.preventDefault();

                window.Morpheus.dialog.confirm(openButtonWheel.dataset.confirmation, async result => {
                    if (result.isConfirmed) {
                        resetWheel();

                        try {
                            const response = await fetch(openButtonWheel.getAttribute('href'), {
                                headers: {
                                    'Accept': 'application/json'
                                }
                            });
                            const data = await response.json();

                            if (data.success) {
                                const stop = wheel.getRandomForSegment(data.index + 1);
                                wheel.successMessage = data.message;
                                wheel.animation.stopAngle = stop;
                                wheel.startAnimation();
                            } else {
                                window.Morpheus.dialog.error(data.message);
                            }
                        } catch (err) {
                            const error = typeof (err.error) === 'object' ? err.error : err;
                            const message = error.error ?? error.message ?? null;

                            window.Morpheus.dialog.error(message);
                        }
                    }
                })
            });
        }
    }
}

function getElementMargin(element) {
    const computedStyle = window.getComputedStyle(element)
    const margin = computedStyle.marginLeft

    return parseFloat(margin)
}

async function createCase() {
    const lootboxCase = document.getElementById('lootbox-case');
    if (lootboxCase) {
        const response = await fetch(`${window.Morpheus.basePath}lootboxes/${lootboxCase.dataset.slug}/items`);
        const data = await response.json();

        const container = lootboxCase.querySelector('[data-lootbox-case-container]');
        const w = lootboxCase.querySelector('.lootbox-case__window');

        const clonedAppendItems = [...container.children].map(child => child.cloneNode(true));
        const clonedPrependItems = [...container.children].map(child => child.cloneNode(true));

        const firstItem = container.children[0]

        clonedAppendItems.forEach(item => container.appendChild(item));
        clonedPrependItems.reverse().forEach(item => container.prepend(item));

        const margin = getElementMargin(firstItem) ?? 0;
        const startTransformPosition = (w.offsetWidth / 2) - firstItem.getBoundingClientRect().left - (margin * clonedAppendItems.length * 2);

        container.dataset.initTranslatePosition = startTransformPosition
        container.style.transform = `translateX(${startTransformPosition}px)`

        const openButtonCase = document.querySelector('[data-open-case]');
        if (openButtonCase) {
            openButtonCase.removeAttribute('disabled');
            openButtonCase.addEventListener('click', (e) => {
                e.preventDefault();

                window.Morpheus.dialog.confirm(openButtonCase.dataset.confirmation, async result => {
                    if (result.isConfirmed) {
                        try {
                            const response = await fetch(openButtonCase.getAttribute('href'), {
                                headers: {
                                    'Accept': 'application/json'
                                }
                            });
                            const data = await response.json();

                            if (data.success) {
                                const spin = new SpinCase(data.index, function () {
                                    window.Morpheus.dialog.success(data.message);
                                });
                                spin.start();
                            } else {
                                window.Morpheus.dialog.error(data.message);
                            }
                        } catch (err) {
                            const error = typeof (err.error) === 'object' ? err.error.message : err.message;

                            window.Morpheus.dialog.error(error);
                        }
                    }
                })
            });
        }
    }
}

class SpinCase {
    constructor(wonIndex, complete) {
        this.el = document.querySelector('[data-lootbox-case-container]');
        this.items = Array.from(this.el.querySelectorAll('[data-lootbox-case-item]'));
        this.position = 0;
        this.numItems = this.items.length;
        this.itemWidth = this.items[0].offsetWidth + 10;
        this.spinTime = this.el.dataset.speed ?? 5000;
        this.running = false;
        this.resultId = wonIndex;
        this.loops = 1;
        this.totalWidth = this.numItems * this.itemWidth;
        this.complete = complete;
    }

    start() {
        if (this.running) return;

        const random = Math.floor(Math.random() * (this.itemWidth / 2));
        const resultOffset = (this.resultId * this.itemWidth + Number(this.el.dataset.initTranslatePosition) + random) - this.itemWidth;
        this.totalDistance = (this.loops * this.totalWidth) + resultOffset;

        this.running = true;
        this.startTime = performance.now();

        const animate = (rafTime) => {
            this.update(rafTime);
            if (this.running) {
                requestAnimationFrame(animate);
            }
        };

        requestAnimationFrame(animate);
    }

    update(rafTime) {
        const deltaTime = rafTime - this.startTime;
        if (deltaTime >= this.spinTime) {
            this.running = false;
            this.complete();
            return;
        }

        const t = this.easeOutSine(deltaTime, 0, 1, this.spinTime);
        this.position = Math.round(t * this.totalDistance);
        const translateX = (this.position % this.totalWidth) + Number(this.el.dataset.initTranslatePosition);

        this.el.style.transform = `translateX(-${translateX}px)`;
    }

    easeOutSine(t, b, c, d) {
        return c * Math.sin((t / d) * (Math.PI / 2)) + b;
    }
}

DOMLoaded(() => {
    createWheel();
    createCase();
})