function openPopupCenter({url, title, w, h}) {
    const dualScreenLeft = window.screenLeft !==  undefined ? window.screenLeft : window.screenX;
    const dualScreenTop = window.screenTop !==  undefined   ? window.screenTop  : window.screenY;

    const width = window.innerWidth ? window.innerWidth : document.documentElement.clientWidth ? document.documentElement.clientWidth : screen.width;
    const height = window.innerHeight ? window.innerHeight : document.documentElement.clientHeight ? document.documentElement.clientHeight : screen.height;

    const systemZoom = width / window.screen.availWidth;
    const left = (width - w) / 2 / systemZoom + dualScreenLeft
    const top = (height - h) / 2 / systemZoom + dualScreenTop
    const newWindow = window.open(url, title,
        `
      scrollbars=yes,
      width=${w / systemZoom},
      height=${h / systemZoom},
      top=${top},
      left=${left}
      `
    )
    if (window.focus) newWindow.focus();
}

function copyText(containerid) {
    if (document.selection) {
        const range = document.body.createTextRange();
        range.moveToElementText(document.getElementById(containerid));
        range.select();
    } else if (window.getSelection) {
        const range = document.createRange();
        range.selectNode(document.getElementById(containerid));
        window.getSelection().removeAllRanges();
        window.getSelection().addRange(range);
    }}

function printf() {
    const args = Array.from(arguments)
    let str = args.shift()

    return args.reduce((str, val, i) => {
        let reg = new RegExp(`\\{${i}\\}`, 'g')
        return str.replace(reg, val)
    }, str)
}

function findNextElementWithClass(element, className) {
    let nextElement = element.nextElementSibling;

    while (nextElement) {
        if (nextElement.classList.contains(className)) {
            return nextElement;
        }
        nextElement = nextElement.nextElementSibling;
    }

    return null;
}

function DOMLoaded(callback) {
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', callback);
    } else {
        callback();
    }
}

class Form {
    async submit(el) {
        const method = el.getAttribute('method') ?? 'POST';
        const formData = new FormData(el);
        const data = new FormData();

        for (const pair of formData.entries()) {
            const key = pair[0];
            const value = pair[1];

            if (value instanceof File) {
                data.append(`${key}[]`, value);
            } else {
                data.append(key, value);
            }
        }

        const response = await fetch(el.getAttribute('action'), {
            method,
            headers: {
                accept: 'application/json'
            },
            body: data
        });
        const json = await response.json();

        return new Promise((resolve, reject) => {
            this.clearErrors(el);

            if (json.success) {
                resolve(json);
            } else {
                if (json.errors) {
                    for (const name in json.errors) {
                        const text = json.errors[name];

                        const field = el.querySelector(`[name="${name}"]`);
                        if (field) {
                            field.classList.add('is-invalid');

                            const feedback = findNextElementWithClass(field, 'invalid-feedback');
                            if (!feedback) {
                                field.insertAdjacentHTML('afterend', `<div class="invalid-feedback">${text}</div>`);
                            } else {
                                feedback.innerHTML = text;
                            }
                        }
                    }

                    reject(json);
                } else {
                    reject(json);
                }
            }
        });
    }

    clearErrors(form) {
        const fields = form.querySelectorAll('.is-invalid');
        fields.forEach(field => field.classList.remove('is-invalid'));
    }

    button(button, status) {
        switch (status) {
            case 'loading':
                const html = button.innerHTML;

                button.disabled = true;
                button.dataset.label = html;
                button.innerHTML = `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span>`;
                break;
            case 'reset':
                const label = button.dataset.label;

                button.removeAttribute('disabled');
                delete button.dataset.label;
                button.innerHTML = label;

                break;
        }
    }
}

class Dialog {
    async success(message, callback) {
        const result = await Swal.fire({
            icon: 'success',
            html: message
        });

        if (callback) {
            callback(result);
        }
    }

    async error(message, callback) {
        const result = await Swal.fire({
            icon: 'error',
            html: message
        });

        if (callback) {
            callback(result);
        }
    }

    async confirm(message, callback) {
        const result = await Swal.fire({
            title: 'Are you sure?',
            html: message,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#3085d6',
            cancelButtonColor: '#d33',
            confirmButtonText: 'Yes'
        });

        callback(result);
    }

    async prompt(message, description, callback) {
        const result = await Swal.fire({
            title: message,
            text: description,
            input: 'text',
            inputAttributes: {
                autocapitalize: 'off'
            },
            showCancelButton: true,
            confirmButtonText: 'Enviar',
            showLoaderOnConfirm: true,
            allowOutsideClick: () => !Swal.isLoading()
        });

        callback(result);
    }
}

class Auth {
    async login(el) {
        const dialog = new Dialog();
        const form = new Form();

        form.button(el.querySelector('button[type=submit]'), 'loading');

        try {
            const response = await form.submit(el);

            if (response.partial) {
                //loadPartial(data.partial);
            }

            if (response.redirect) {
                window.location = response.redirect;
            }
        } catch (err) {
            if (err.message) {
                dialog.error(err.message);
            }
        } finally {
            form.button(el.querySelector('button[type=submit]'), 'reset');
        }
    }
}

function runFormRequest(ctx) {
    const formRequest = ctx.querySelectorAll('[data-form-request]');
    formRequest.forEach(el => {
        el.addEventListener('submit', e => {
            e.preventDefault();

            async function sendRequest() {
                window.Morpheus.form.button(el.querySelector('button[type=submit]'), 'loading');

                function processResponse(response) {
                    if (response.redirect) {
                        window.location = response.redirect;
                    }
                }

                function processResponseError(err) {
                    if (err.redirect) {
                        window.location = err.redirect;
                    }
                }

                try {
                    const response = await window.Morpheus.form.submit(el);

                    el.reset();

                    if (response.message) {
                        window.Morpheus.dialog.success(response.message, result => processResponse(response));
                    } else {
                        processResponse(response);
                    }
                } catch (err) {
                    const error = typeof(err.error) === 'object' ? err.error : err;
                    const message = error.error ?? error.message ?? null;

                    if (message) {
                        window.Morpheus.dialog.error(message, result => processResponseError(error));
                    } else {
                        processResponseError(error);
                    }
                } finally {
                    window.Morpheus.form.button(el.querySelector('button[type=submit]'), 'reset');
                }
            }

            if (el.dataset?.confirmation) {
                window.Morpheus.dialog.confirm(el.dataset.confirmation, result => {
                    if (result.isConfirmed) sendRequest();
                });
            } else if (el.dataset?.requiredPid) {
                window.Morpheus.dialog.prompt(el.dataset.requiredPid ?? 'Informe seu personal ID', null, result => {
                    if (result.isConfirmed) {
                        el.querySelector('input[name="pid"]').value = result.value;
                        sendRequest();
                    }
                });
            } else if (el.dataset?.requiredToken) {
                window.Morpheus.dialog.prompt(el.dataset.requiredToken ?? 'Informe seu Token', null, result => {
                    if (result.isConfirmed) sendRequest();
                });
            } else {
                sendRequest();
            }
        });
    });
}

function runActionRequest(ctx) {
    const actionRequest = ctx.querySelectorAll('[data-action-request]');
    actionRequest.forEach(el => {
        el.addEventListener('click', e => {
            e.preventDefault();

            const actionUrl = el.getAttribute('href');

            async function sendRequest(url) {
                function processResponse(response) {
                    if (response.redirect) {
                        window.location = response.redirect;
                    }
                }

                function processResponseError(err) {
                    if (err.redirect) {
                        window.location = err.redirect;
                    }
                }

                const response = await fetch(url, {
                    headers: {
                        accept: 'application/json'
                    }
                });
                const json = await response.json();

                if (json.success) {
                    if (json.message) {
                        await window.Morpheus.dialog.success(json.message, () => processResponse(json));
                    } else {
                        processResponse(json);
                    }
                } else {
                    if (json.confirm) {
                        await window.Morpheus.dialog.confirm(json.message, async result => {
                            if (result.isConfirmed) {
                                const acceptTerm = actionUrl.includes('?') ? '&accept' : '?accept';
                                const completeUrl = `${actionUrl}${acceptTerm}`;
                                sendRequest(completeUrl);
                            }
                        });
                    } else if (json.required_token) {
                        await window.Morpheus.dialog.prompt(json.token_message, null, async result => {
                            if (result.isConfirmed) {
                                const tokenTerm = actionUrl.includes('?') ? '&token' : '?token';
                                const completeUrl = `${actionUrl}${tokenTerm}=${result.value}`;
                                sendRequest(completeUrl);
                            }
                        });
                    } else if (json.required_pid) {
                        await window.Morpheus.dialog.prompt(json.pid_message, null, async result => {
                            if (result.isConfirmed) {
                                const pidTerm = actionUrl.includes('?') ? '&pid' : '?pid';
                                const completeUrl = `${actionUrl}${pidTerm}=${result.value}`;
                                sendRequest(completeUrl);
                            }
                        });
                    } else {
                        const error = typeof (json.error) === 'object' ? json.error : json;
                        const message = error.error ?? error.message ?? null;

                        if (message) {
                            await window.Morpheus.dialog.error(message, () => processResponseError(error));
                        } else {
                            processResponseError(error);
                        }
                    }
                }
            }

            if (el.dataset?.confirmation) {
                window.Morpheus.dialog.confirm(el.dataset.confirmation, result => {
                    if (result.isConfirmed) sendRequest(actionUrl);
                });
            } else if (el.dataset?.requiredPid) {
                window.Morpheus.dialog.prompt(el.dataset.requiredPid ?? 'Informe seu personal ID', null, result => {
                    if (result.isConfirmed) sendRequest(actionUrl);
                });
            } else if (el.dataset?.requiredToken) {
                window.Morpheus.dialog.prompt(el.dataset.requiredToken ?? 'Informe seu Token', null, result => {
                    if (result.isConfirmed) sendRequest(actionUrl);
                });
            } else {
                sendRequest(actionUrl);
            }
        });
    })
}

(function () {
    const dialog = new Dialog();
    const auth = new Auth();
    const form = new Form();

    Object.assign(window.Morpheus, {
        dialog,
        auth,
        form
    });

    runFormRequest(document);
    runActionRequest(document);

    const formAuth = document.querySelectorAll('[data-form-request-auth]');
    formAuth.forEach(el => {
        el.addEventListener('submit', e => {
            auth.login(el);

            e.preventDefault();
        });
    })

    const openNewWindow = document.querySelectorAll('[data-open-new-window]');
    openNewWindow.forEach(el => {
        el.addEventListener('click', e => {
            e.preventDefault();
            openPopupCenter({
                url: el.getAttribute('href'),
                w: 500,
                h: 500
            })
            window.open(el.getAttribute('href'));
        });
    });

    const copyTextElements = document.querySelectorAll('[data-copy-text]');
    copyTextElements.forEach(el => {
        el.addEventListener('click', e => {
            e.preventDefault();
            copyText(el.getAttribute('data-copy-text'));
            document.execCommand('copy');
        });
    });

    const inputsRange = document.querySelectorAll('.form-range-control');
    inputsRange.forEach(el => {
        const input = el.querySelector('.form-range');

        input.addEventListener('input', (e) => {
            const inputContent = el.querySelector('.form-range-text-content');
            inputContent.textContent = e.target.value;
        });
    });

    const tooltips = document.querySelectorAll('[data-tooltip]');
    tooltips.forEach(el => {
        const tooltipUrl = el.dataset.tooltipUrl;
        if (tooltipUrl) {
            tippy(el, {
                followCursor: true,
                animateFill: false,
                animation: 'fade',
                maxWidth: 'auto',
                content: 'Carregando...',
                onShow(instance) {
                    fetch(tooltipUrl)
                        .then(response => response.text())
                        .then(html => instance.setContent(html));
                }
            })
        }
    });

    const currencyInputs = document.querySelectorAll('[data-currency-input]');
    currencyInputs.forEach(el => {
        new AutoNumeric(el, {
            currencySymbol: 'R$ ',
            decimalCharacter: ',',
            digitGroupSeparator: '.',
            decimalPlaces: 2,
            decimalPlacesShownOnFocus: 2,
            decimalPlacesShownOnBlur: 2,
            minimumValue: '0',
            modifyValueOnWheel: false,
            allowDecimalPadding: true,
            roundingMethod: 'U',
        });
    });

    const loadInModals = document.querySelectorAll('[data-open-modal]');
    loadInModals.forEach(el => {
        el.addEventListener('click', e => {
            e.preventDefault();
            const url = el.getAttribute('href');
            const modalEl = document.getElementById(el.dataset.openModal)
            const modal = new bootstrap.Modal(modalEl)
            modal.show();

            fetch(url)
                .then(response => response.text())
                .then(html => {
                    const dialog = modalEl.querySelector('.modal-dialog');
                    dialog.innerHTML = html;

                    runFormRequest(dialog)
                });
        });
    });

    const refreshRecaptchaElements = document.querySelectorAll('[data-refresh-recaptcha]')
    refreshRecaptchaElements.forEach(el => {
        el.addEventListener('click', e => {
            e.preventDefault();
            const container = el.closest('[data-recaptcha]');
            const image = container.querySelector('[data-recaptcha-image]');

            image.setAttribute('src', `${window.Morpheus.basePath}captcha?t=${new Date().getTime()}`);
        });
    })
})();